<?php
// classes/GameEngine.php
// Version Multijoueur SQL 1v1

class Entity {
    public $id;
    public $user_id; // Pour savoir à qui est l'unité
    public $type;
    public $x;
    public $y;
    public $hp;
    public $maxHp;
    public $energy;
    public $hidden = false;
    public $name;
    public $can_act; // Si l'unité a déjà bougé ce tour

    public function __construct($row, $isEnemy, $myId) {
        $this->id = $row['id'];
        $this->user_id = $row['user_id'];
        $this->type = ($row['user_id'] == $myId) ? 'ship' : 'enemy'; // Si c'est à moi = ship, sinon = enemy
        
        // Mapping DB -> Jeu
        $this->x = (int)$row['coord_x'];
        $this->y = (int)$row['coord_y'];
        $this->hp = (int)$row['current_hp'];
        $this->maxHp = 100; // Valeur par défaut ou à récupérer d'une join sur table vaisseaux
        $this->energy = (int)$row['energy'];
        $this->name = $row['type_ship'] ?? 'Vaisseau';
        
        // Gestion drones (simplifiée pour l'exemple, à adapter selon ta colonne type)
        if($row['type_ship'] == 'Drone') $this->type = 'drone';
    }
}

class GameManager {
    private $db;
    private $userId;
    private $opponentId;
    private $gridSize = 10;

    public function __construct($dbLink, $userId) {
        $this->db = $dbLink;
        $this->userId = $userId;
        
        // Récupérer l'ID de l'adversaire
        $query = mysqli_query($this->db, "SELECT opponent_id FROM user_profiles WHERE user_id = $this->userId");
        $data = mysqli_fetch_assoc($query);
        $this->opponentId = $data['opponent_id'];
    }

    // --- ACTIONS ---

    public function move($shipId, $x, $y) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Attendez votre tour."];
        if (!$this->isValidPos($x, $y)) return ['success' => false, 'msg' => "Hors zone."];

        // Vérif appartenance
        if (!$this->ownsShip($shipId)) return ['success' => false, 'msg' => "Ce n'est pas votre unité."];
        
        // Vérif case occupée
        if ($this->isOccupied($x, $y)) return ['success' => false, 'msg' => "Case occupée."];

        // Vérif distance (Récupérer pos actuelle depuis DB)
        $q = mysqli_query($this->db, "SELECT coord_x, coord_y FROM user_fleets WHERE id = $shipId");
        $curr = mysqli_fetch_assoc($q);
        $dist = abs($curr['coord_x'] - $x) + abs($curr['coord_y'] - $y);
        
        if ($dist > 1) return ['success' => false, 'msg' => "Trop loin."];

        // Update DB
        mysqli_query($this->db, "UPDATE user_fleets SET coord_x = $x, coord_y = $y WHERE id = $shipId");
        
        return ['success' => true, 'msg' => "Déplacement confirmé."];
    }

    public function attack($attackerId, $targetId) {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Pas votre tour."];
        
        // Récup stats attaquant
        $qAtk = mysqli_query($this->db, "SELECT energy FROM user_fleets WHERE id = $attackerId AND user_id = $this->userId");
        $atk = mysqli_fetch_assoc($qAtk);
        
        if (!$atk) return ['success' => false, 'msg' => "Attaquant invalide."];
        if ($atk['energy'] < 10) return ['success' => false, 'msg' => "Manque d'énergie."];

        // Appliquer dégâts
        $dmg = rand(15, 25);
        mysqli_query($this->db, "UPDATE user_fleets SET current_hp = current_hp - $dmg WHERE id = $targetId");
        mysqli_query($this->db, "UPDATE user_fleets SET energy = energy - 10 WHERE id = $attackerId");

        // Vérifier destruction
        $qTarget = mysqli_query($this->db, "SELECT current_hp FROM user_fleets WHERE id = $targetId");
        $tgt = mysqli_fetch_assoc($qTarget);
        
        $msg = "Touché (-$dmg)!";
        if ($tgt['current_hp'] <= 0) {
            mysqli_query($this->db, "DELETE FROM user_fleets WHERE id = $targetId");
            $msg = "Cible détruite !";
            $this->checkWinCondition();
        }

        return ['success' => true, 'msg' => $msg];
    }

    public function endTurn() {
        if (!$this->isMyTurn()) return ['success' => false, 'msg' => "Ce n'est pas votre tour."];

        // 1. Passer la main
        mysqli_query($this->db, "UPDATE user_profiles SET is_my_turn = 0 WHERE user_id = $this->userId");
        mysqli_query($this->db, "UPDATE user_profiles SET is_my_turn = 1 WHERE user_id = $this->opponentId");

        // 2. Regen Energy de mon adversaire (pour qu'il soit prêt à jouer)
        mysqli_query($this->db, "UPDATE user_fleets SET energy = LEAST(energy + 10, 100) WHERE user_id = $this->opponentId");

        return ['success' => true];
    }

    // --- LECTURE ETAT ---

    public function getGameState() {
        $entities = [];
        
        // Récupérer TOUTES les flottes (Moi + Adversaire)
        $sql = "SELECT * FROM user_fleets WHERE user_id = $this->userId OR user_id = $this->opponentId";
        $res = mysqli_query($this->db, $sql);

        // Récupérer ma flotte pour calcul brouillard
        $myShips = [];
        $allRows = [];
        while($row = mysqli_fetch_assoc($res)) {
            $allRows[] = $row;
            if($row['user_id'] == $this->userId) $myShips[] = $row;
        }

        foreach($allRows as $row) {
            $isEnemy = ($row['user_id'] != $this->userId);
            $ent = new Entity($row, $isEnemy, $this->userId);
            
            // Logique de Brouillard de Guerre
            if ($isEnemy) {
                $ent->hidden = true;
                foreach ($myShips as $me) {
                    $dist = abs($me['coord_x'] - $ent->x) + abs($me['coord_y'] - $ent->y);
                    // Portée de vue : 3 cases (hardcodé pour l'instant)
                    if ($dist <= 3) {
                        $ent->hidden = false;
                        break;
                    }
                }
            } else {
                $ent->hidden = false;
            }

            // On n'envoie pas les ennemis cachés pour éviter la triche via inspecteur
            // SI on veut tricher "gentiment" ou debug, on envoie tout avec flag hidden
            // Ici, j'envoie tout avec le flag pour que le JS gère l'affichage fantôme ou non
            $entities[] = $ent;
        }

        return [
            'entities' => $entities,
            'turn' => $this->isMyTurn() ? 'player' : 'enemy',
            'win' => $this->checkWinCondition(false)
        ];
    }

    // --- HELPERS ---

    private function isMyTurn() {
        $q = mysqli_query($this->db, "SELECT is_my_turn FROM user_profiles WHERE user_id = $this->userId");
        $row = mysqli_fetch_assoc($q);
        return $row && $row['is_my_turn'] == 1;
    }

    private function ownsShip($shipId) {
        $q = mysqli_query($this->db, "SELECT id FROM user_fleets WHERE id = $shipId AND user_id = $this->userId");
        return mysqli_num_rows($q) > 0;
    }

    private function isOccupied($x, $y) {
        $q = mysqli_query($this->db, "SELECT id FROM user_fleets WHERE coord_x = $x AND coord_y = $y AND (user_id = $this->userId OR user_id = $this->opponentId)");
        return mysqli_num_rows($q) > 0;
    }

    private function isValidPos($x, $y) {
        return ($x >= 0 && $x < $this->gridSize && $y >= 0 && $y < $this->gridSize);
    }
    
    private function checkWinCondition($exec = true) {
        // Compter les vaisseaux restants
        $qMe = mysqli_query($this->db, "SELECT COUNT(*) as c FROM user_fleets WHERE user_id = $this->userId");
        $qOp = mysqli_query($this->db, "SELECT COUNT(*) as c FROM user_fleets WHERE user_id = $this->opponentId");
        
        $myCount = mysqli_fetch_assoc($qMe)['c'];
        $opCount = mysqli_fetch_assoc($qOp)['c'];

        if ($myCount == 0) return 'LOSE';
        if ($opCount == 0) return 'WIN';
        return false;
    }
    
    public function reset() {
        // En 1v1 réel, reset est complexe. Ici on ne fait rien ou on redirige lobby.
        // Pour le test, on pourrait remettre les positions initiales.
    }
}
?>