<?php
// system_audit.php
// Script de diagnostic complet pour le système de combat 1v1
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// --- CONFIGURATION ---
// Liste des fichiers critiques attendus
$required_files = [
    'includes/db.php',
    'api.php',
    'univers.php',
    'GameEngine.php' // ou classes/GameEngine.php
];

// Liste des tables et colonnes OBLIGATOIRES pour que le code GameEngine fonctionne
// Format: 'table' => ['colonne1', 'colonne2'...]
$db_schema_requirements = [
    'user_profiles' => ['user_id', 'opponent_id', 'is_my_turn'],
    'user_fleets'   => ['id', 'user_id', 'coord_x', 'coord_y', 'current_hp', 'energy', 'type_ship']
];

// --- STYLES ---
echo "<style>
    body { font-family: monospace; background: #1a202c; color: #cbd5e0; padding: 20px; }
    .card { background: #2d3748; padding: 15px; margin-bottom: 20px; border-radius: 8px; border-left: 5px solid #4a5568; }
    .ok { border-left-color: #48bb78; }
    .fail { border-left-color: #e53e3e; }
    .warn { border-left-color: #ecc94b; }
    h2 { margin-top: 0; border-bottom: 1px solid #4a5568; padding-bottom: 10px; }
    .status { float: right; font-weight: bold; }
    .status.ok { color: #48bb78; } .status.fail { color: #e53e3e; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #4a5568; padding: 8px; text-align: left; }
    code { background: #171923; padding: 2px 5px; border-radius: 3px; color: #ed8936; }
</style>";

echo "<h1>🔍 Audit Système : Combat Spatial 1v1</h1>";

// ==================================================================
// ETAPE 1 : FICHIERS & ENVIRONNEMENT
// ==================================================================
echo "<div class='card'>";
echo "<h2>1. Vérification des Fichiers & PHP</h2>";

// Check PHP
echo "<div>Version PHP : <b>" . phpversion() . "</b> ";
if (version_compare(phpversion(), '7.0.0', '<')) {
    echo "<span class='status fail'>[OBSOLÈTE]</span> - Le code moderne nécessite PHP 7+";
} else {
    echo "<span class='status ok'>[OK]</span>";
}
echo "</div>";

// Check Session
if (session_status() === PHP_SESSION_ACTIVE) {
    echo "<div>Sessions PHP : <span class='status ok'>[ACTIVES]</span></div>";
    if (isset($_SESSION['user_id'])) {
        echo "<div>Utilisateur Connecté : ID <b>" . $_SESSION['user_id'] . "</b> <span class='status ok'>[OK]</span></div>";
        $my_id = $_SESSION['user_id'];
    } else {
        echo "<div>Utilisateur Connecté : <span class='status fail'>[NON]</span> (Variable \$_SESSION['user_id'] vide)</div>";
        echo "<small>⚠️ Certains tests de données seront ignorés.</small>";
        $my_id = null;
    }
} else {
    echo "<div>Sessions PHP : <span class='status fail'>[INACTIVES]</span> (session_start() manquant ?)</div>";
}

// Check Files
echo "<table><tr><th>Fichier</th><th>Chemin</th><th>État</th></tr>";
$all_files_ok = true;
foreach ($required_files as $file) {
    // Vérification un peu souple (racine ou dossier classes/)
    $exists = file_exists($file);
    if (!$exists && $file == 'GameEngine.php') {
        if (file_exists('classes/GameEngine.php')) { $file = 'classes/GameEngine.php'; $exists = true; }
    }
    
    $status = $exists ? "<span class='status ok'>PRÉSENT</span>" : "<span class='status fail'>ABSENT</span>";
    if (!$exists) $all_files_ok = false;
    echo "<tr><td>$file</td><td>" . realpath($file) . "</td><td>$status</td></tr>";
}
echo "</table>";
echo "</div>";

// ==================================================================
// ETAPE 2 : BASE DE DONNÉES & SCHÉMA
// ==================================================================
echo "<div class='card'>";
echo "<h2>2. Connexion & Structure BDD</h2>";

$db_path = 'includes/db.php';
if (!file_exists($db_path)) {
    echo "<h3 class='status fail'>Arrêt critique : includes/db.php introuvable.</h3></div>";
    exit;
}

require_once $db_path;

if (isset($link) && $link instanceof mysqli) {
    echo "<div>Connexion MySQL : <span class='status ok'>[RÉUSSIE]</span></div>";
    echo "<div>Host info : " . mysqli_get_host_info($link) . "</div>";
    
    // ANALYSE DU SCHÉMA
    echo "<h3>Analyse des Colonnes (GameEngine requirements)</h3>";
    echo "<table><tr><th>Table</th><th>Colonne Requise</th><th>État</th><th>Type Détecté</th></tr>";
    
    foreach ($db_schema_requirements as $table => $cols) {
        // Vérif si table existe
        $check_table = mysqli_query($link, "SHOW TABLES LIKE '$table'");
        if (mysqli_num_rows($check_table) == 0) {
            echo "<tr><td><b>$table</b></td><td colspan='3' class='status fail'>TABLE MANQUANTE !</td></tr>";
            continue;
        }

        // Récup des colonnes
        $res_cols = mysqli_query($link, "SHOW COLUMNS FROM $table");
        $existing_cols = [];
        while ($r = mysqli_fetch_assoc($res_cols)) {
            $existing_cols[$r['Field']] = $r['Type'];
        }

        foreach ($cols as $req_col) {
            $found = array_key_exists($req_col, $existing_cols);
            $style = $found ? "ok" : "fail";
            $msg = $found ? "OK" : "MANQUANTE";
            $type = $found ? $existing_cols[$req_col] : "-";
            
            // Warning pour les types numériques
            if ($found && ($req_col == 'coord_x' || $req_col == 'current_hp') && strpos($type, 'int') === false) {
                 $msg = "OK (⚠️ Type $type suspect)";
                 $style = "warn";
            }

            echo "<tr><td>$table</td><td>$req_col</td><td><span class='status $style'>$msg</span></td><td>$type</td></tr>";
        }
    }
    echo "</table>";

} else {
    echo "<h3 class='status fail'>Échec Connexion : La variable \$link n'existe pas ou n'est pas un objet mysqli.</h3>";
    echo "<pre>Contenu de db.php chargé mais connexion invalide.</pre>";
    exit; // On ne peut pas aller plus loin
}
echo "</div>";

// ==================================================================
// ETAPE 3 : LOGIQUE DE JEU & DONNÉES
// ==================================================================
if ($my_id) {
    echo "<div class='card'>";
    echo "<h2>3. Logique & Cohérence des Données (User $my_id)</h2>";

    // 1. Profil Utilisateur
    $q_prof = mysqli_query($link, "SELECT * FROM user_profiles WHERE user_id = $my_id");
    $prof = mysqli_fetch_assoc($q_prof);

    if ($prof) {
        echo "<h3>État du Profil</h3>";
        $opp_id = $prof['opponent_id'];
        $turn = $prof['is_my_turn'];
        
        echo "<ul>";
        echo "<li><b>Adversaire (opponent_id) :</b> " . ($opp_id ? "$opp_id <span class='status ok'>[OK]</span>" : "NULL <span class='status fail'>[PAS D'ADVERSAIRE]</span>") . "</li>";
        echo "<li><b>Mon tour (is_my_turn) :</b> " . ($turn ? "OUI (1)" : "NON (0)") . "</li>";
        echo "</ul>";

        if (!$opp_id) {
            echo "<div class='status fail'>⚠️ CRITIQUE : Le jeu ne peut pas démarrer sans adversaire. Passez par le Lobby ou forcez une valeur dans la BDD.</div>";
        }
    } else {
        echo "<div class='status fail'>Profil introuvable dans user_profiles pour l'ID $my_id</div>";
    }

    // 2. Flotte
    echo "<h3>Analyse de la Flotte</h3>";
    $q_fleet = mysqli_query($link, "SELECT * FROM user_fleets WHERE user_id = $my_id");
    $count = mysqli_num_rows($q_fleet);
    
    echo "<div>Nombre de vaisseaux : <b>$count</b> " . ($count > 0 ? "<span class='status ok'>[OK]</span>" : "<span class='status fail'>[VIDE]</span>") . "</div>";

    if ($count > 0) {
        echo "<table><tr><th>ID</th><th>Type</th><th>HP</th><th>Énergie</th><th>Pos (X,Y)</th><th>Valide ?</th></tr>";
        while ($ship = mysqli_fetch_assoc($q_fleet)) {
            $x = $ship['coord_x'];
            $y = $ship['coord_y'];
            $hp = $ship['current_hp'];
            
            $valid = true;
            $errs = [];
            if ($x < 0 || $x > 9 || $y < 0 || $y > 9) { $valid = false; $errs[] = "Hors Map"; }
            if ($hp <= 0) { $valid = false; $errs[] = "Détruit (0 HP)"; }
            
            $status_html = $valid ? "<span class='status ok'>OK</span>" : "<span class='status fail'>" . implode(', ', $errs) . "</span>";
            
            echo "<tr>
                <td>{$ship['id']}</td>
                <td>{$ship['type_ship']}</td>
                <td>$hp</td>
                <td>{$ship['energy']}</td>
                <td>($x, $y)</td>
                <td>$status_html</td>
            </tr>";
        }
        echo "</table>";
    }

    // 3. Test Adversaire (si existe)
    if (isset($opp_id) && $opp_id) {
        $q_opp = mysqli_query($link, "SELECT COUNT(*) as c FROM user_fleets WHERE user_id = $opp_id");
        $d_opp = mysqli_fetch_assoc($q_opp);
        echo "<h3>Analyse Adversaire (ID $opp_id)</h3>";
        echo "<div>Vaisseaux ennemis détectés : <b>" . $d_opp['c'] . "</b> " . ($d_opp['c'] > 0 ? "<span class='status ok'>[OK]</span>" : "<span class='status warn'>[AUCUN] - Victoire automatique ?</span>") . "</div>";
    }

    echo "</div>";
}
?>